// -*- Mode: C++ -*-

// ----------------------------------------------------------------------------
//  m_stream.cpp  2002/01/27
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  CN[h
// ----------------------------------------------------------------------------

#include "m_stream.h"

#include <stdio.h>
#include <stdlib.h>
#include "util.h"

namespace m_stream
{


// ----------------------------------------------------------------------------
//  萔
// ----------------------------------------------------------------------------

const int BUF_SIZE = 16384;


// ----------------------------------------------------------------------------
//  I[vς݂̃t@C(nh)̓o
// ----------------------------------------------------------------------------

class StreamWinHandle : public Stream
{
public:
  StreamWinHandle(void) { mFH = INVALID_HANDLE_VALUE; }
  virtual ~StreamWinHandle() { Close(); }

  // StreamWinHandle ł͈ȉR inDataLen ͖܂
  int OpenIn(const void *inFH, DWORD inDataLen);
  int OpenOut(const void *inFH, DWORD inDataLen);
  int OpenInOut(const void *inFH, DWORD inDataLen);

  void Close(void);

  int GetByte(void);
  int PutByte(int inC);

  int GetBytes(void *outBuf, DWORD inLen);
  int PutBytes(const void *inBuf, DWORD inLen);
  int PutBytes(Stream *inSrc, DWORD inLen);

  int Seek(DWORD inPos);
  DWORD Tell(void);
  DWORD GetLen(void);

  int IsEndOfStream(void);

protected:
  HANDLE  mFH;

  void InitBuffer(void)
  {
    mBufPos = mBufEnd = 0;
  }

  DWORD FillBuffer(void)
  {
    mBufPos = 0;
    ReadFile(mFH, mBuf, BUF_SIZE, &mBufEnd, NULL);
    return mBufEnd;
  }

  DWORD FlushBuffer(void)
  {
    DWORD  theWB;
    if (mBufPos)
    {
      WriteFile(mFH, mBuf, mBufPos, &theWB, NULL);
      mBufPos = 0;
    }
    mBufEnd = BUF_SIZE;
    return theWB;
  }

  BYTE   mBuf[BUF_SIZE];
  DWORD  mBufPos;
  DWORD  mBufEnd;

  enum { TYPE_IN, TYPE_OUT, TYPE_IO }  mType;
};

int StreamWinHandle::OpenIn(const void *inFH, DWORD inDataLen)
{
  Close();
  InitBuffer();
  mType = TYPE_IN;
  mFH = (HANDLE)inFH;
  if (INVALID_HANDLE_VALUE == mFH)
    return -1;
  else
    return 0;
}

int StreamWinHandle::OpenOut(const void *inFH, DWORD inDataLen)
{
  Close();
  InitBuffer();
  mType = TYPE_OUT;
  mFH = (HANDLE)inFH;
  if (INVALID_HANDLE_VALUE == mFH)
    return -1;
  else
    return 0;
}

int StreamWinHandle::OpenInOut(const void *inFH, DWORD inDataLen)
{
  Close();
  return -1;
}

void StreamWinHandle::Close(void)
{
  if (INVALID_HANDLE_VALUE != mFH)
  {
    if (TYPE_OUT == mType)
      FlushBuffer();
    mFH = INVALID_HANDLE_VALUE;
  }
}

int StreamWinHandle::GetByte(void)
{
  if (INVALID_HANDLE_VALUE == mFH)
    return -1;

  if (mBufPos == mBufEnd)
  {
    if (0 == FillBuffer())  // t@CI[H
      return Stream::EOS;
  }

  return mBuf[mBufPos++];
}

int StreamWinHandle::PutByte(int inC)
{
  if (INVALID_HANDLE_VALUE == mFH)
    return -1;

  if (mBufPos == mBufEnd)
    FlushBuffer();

  mBuf[mBufPos++] = (BYTE)inC;

  return 0;
}

int StreamWinHandle::GetBytes(void *outBuf, DWORD inLen)
{
  if (INVALID_HANDLE_VALUE == mFH)
    return -1;

  if (mBufPos == mBufEnd)
  {
    if (0 == FillBuffer())  // t@CI[H
      return Stream::EOS;
  }

  // obt@ǂݍ
  DWORD  theBufRemain = mBufEnd - mBufPos;
  DWORD  theCopyLen = (theBufRemain > inLen) ? inLen : theBufRemain;
  memcpy(outBuf, mBuf + mBufPos, theCopyLen);
  mBufPos += theCopyLen;

  // obt@ɂf[^ł͑ȂH
  inLen -= theCopyLen;
  if (inLen)
  {
    // ȂAړǂݍ
    DWORD  theRB;
    ReadFile(mFH, (BYTE *)outBuf + theCopyLen, inLen, &theRB, NULL);
    theCopyLen += theRB;
    InitBuffer();
  }

  return theCopyLen;
}

int StreamWinHandle::PutBytes(const void *inBuf, DWORD inLen)
{
  if (INVALID_HANDLE_VALUE == mFH)
    return -1;

  if (mBufPos == mBufEnd)
    FlushBuffer();

  DWORD  theBufRemain = mBufEnd - mBufPos;
  DWORD  theCopyLen = (theBufRemain > inLen) ? inLen : theBufRemain;

  memcpy(mBuf + mBufPos, inBuf, theCopyLen);
  mBufPos += theCopyLen;

  inLen -= theCopyLen;
  if (inLen)
  {
    FlushBuffer();

    DWORD  theWB;
    WriteFile(mFH, (BYTE *)inBuf + theCopyLen, inLen, &theWB, NULL);
    theCopyLen += theWB;
  }

  return theCopyLen;
}

int StreamWinHandle::PutBytes(Stream *inSrc, DWORD inLen)
{
  if (INVALID_HANDLE_VALUE == mFH)
    return -1;

  if (mBufPos == mBufEnd)
    FlushBuffer();

  DWORD  theCopyLen, theBufRemain, theReadBytes, theTotalReadBytes = 0;
  while (1)
  {
    // obt@֓ǂݏooCgvZ
    theBufRemain = mBufEnd - mBufPos;
    theCopyLen = (theBufRemain >= inLen) ? inLen : theBufRemain;

    // ǂݏo
    theReadBytes = inSrc->GetBytes(mBuf + mBufPos, theCopyLen);
    mBufPos += theReadBytes;
    inLen -= theReadBytes;
    theTotalReadBytes += theReadBytes;

    // ̓Xg[sH
    if (theReadBytes != theCopyLen)
      return theTotalReadBytes;

    // ܂ǂݏoׂf[^cĂ邩H
    if (inLen)
      FlushBuffer();
  }

  return theTotalReadBytes;
}


int StreamWinHandle::Seek(DWORD inPos)
{
  if (INVALID_HANDLE_VALUE == mFH)
    return -1;
  else
  {
    InitBuffer();
    return SetFilePointer(mFH, inPos, NULL, FILE_BEGIN);
  }
}

DWORD StreamWinHandle::Tell(void)
{
  if (INVALID_HANDLE_VALUE == mFH)
    return DWORD(-1);
  else
    return SetFilePointer(mFH, 0, NULL, FILE_CURRENT) - mBufEnd + mBufPos;
}

DWORD StreamWinHandle::GetLen(void)
{
  if (INVALID_HANDLE_VALUE == mFH)
    return DWORD(-1);
  else
    return GetFileSize(mFH, NULL);
}

int StreamWinHandle::IsEndOfStream(void)
{
  if (INVALID_HANDLE_VALUE == mFH)
    return 0;

  // t@CTCYƌ݂̃t@C̈ʒu
  if (GetLen() == Tell())  // Ȃf[^I[ł
    return 1;
  else
    return 0;
}

Stream *StreamWinHandleFactory::Create(void)
{
  return new StreamWinHandle;
}


// ----------------------------------------------------------------------------
//  t@Co
// ----------------------------------------------------------------------------

class StreamWinFile : public StreamWinHandle
{
public:
  StreamWinFile(void) {}
  ~StreamWinFile() { Close(); }

  int OpenIn(const void *inFileName, DWORD inDataLen);
  int OpenOut(const void *inFileName, DWORD inDataLen);
  int OpenInOut(const void *inFileName, DWORD inDataLen);

  void Close(void);
};


int StreamWinFile::OpenIn(const void *inFileName, DWORD inDataLen)
{
  Close();

  mFH = OpenReadFile((const char *)inFileName);
  if (INVALID_HANDLE_VALUE == mFH)
    return -1;
  else
    return 0;
}

int StreamWinFile::OpenOut(const void *inFileName, DWORD inDataLen)
{
  Close();

  mFH = OpenWriteFile((const char *)inFileName);
  if (INVALID_HANDLE_VALUE == mFH)
    return -1;
  else
    return 0;
}

int StreamWinFile::OpenInOut(const void *inFileName, DWORD inDataLen)
{
  Close();
  return -1;
}

void StreamWinFile::Close(void)
{
  if (INVALID_HANDLE_VALUE != mFH)
  {
    if (TYPE_OUT == mType)
      FlushBuffer();
    CloseHandle(mFH);
    mFH = INVALID_HANDLE_VALUE;
  }
}


Stream *StreamWinFileFactory::Create(void)
{
  return new StreamWinFile;
}


// ----------------------------------------------------------------------------
//  o
// ----------------------------------------------------------------------------

class StreamMemory : public Stream
{
public:
  StreamMemory(void) { mData = NULL; }
  ~StreamMemory() { Close(); }

  int OpenIn(const void *inData, DWORD inDataLen);
  int OpenOut(const void *inData, DWORD inDataLen);
  int OpenInOut(const void *inData, DWORD inDataLen);

  void Close(void);

  int GetByte(void);
  int PutByte(int inC);

  int GetBytes(void *outBuf, DWORD inLen);
  int PutBytes(const void *inBuf, DWORD inLen);
  int PutBytes(Stream *inSrc, DWORD inLen);

  int Seek(DWORD inPos);
  DWORD Tell(void);
  DWORD GetLen(void);

  int IsEndOfStream(void);

private:
  BYTE         *mData;
  DWORD        mDataLen;
  DWORD        mPos;
};

int StreamMemory::OpenIn(const void *inData, DWORD inDataLen)
{
  Close();

  mData = (BYTE *)inData;
  mDataLen = inDataLen;
  mPos = 0;

  return 0;
}

int StreamMemory::OpenOut(const void *inData, DWORD inDataLen)
{
  Close();

  mData = (BYTE *)inData;
  mDataLen = inDataLen;
  mPos = 0;

  return 0;
}

int StreamMemory::OpenInOut(const void *inData, DWORD inDataLen)
{
  Close();
  return -1;
}

void StreamMemory::Close(void)
{
  mData = NULL;
}

int StreamMemory::GetByte(void)
{
  if (NULL == mData || mPos >= mDataLen)
    return -1;
  return mData[mPos++];
}

int StreamMemory::PutByte(int inC)
{
  if (NULL == mData || mPos >= mDataLen)
    return -1;
  mData[mPos++] = inC;
  return 0;
}

int StreamMemory::GetBytes(void *outBuf, DWORD inLen)
{
  if (NULL == mData || mPos >= mDataLen)
    return -1;

  // c̃f[^ȏǂݏoȂ悤ɂ
  if (mPos + inLen > mDataLen)
    inLen = mDataLen - mPos;

  memcpy(outBuf, mData + mPos, inLen);
  mPos += inLen;

  return inLen;
}

int StreamMemory::PutBytes(const void *inData, DWORD inLen)
{
  if (NULL == mData || mPos >= mDataLen)
    return -1;

  if (mPos + inLen > mDataLen)
    inLen = mDataLen - mPos;

  memcpy(mData + mPos, inData, inLen);
  mPos += inLen;

  return inLen;
}

int StreamMemory::PutBytes(Stream *inStrm, DWORD inLen)
{
  if (NULL == mData || mPos >= mDataLen)
    return -1;

  if (mPos + inLen > mDataLen)
    inLen = mDataLen - mPos;

  DWORD  thePos = mPos;
  mPos += inLen;
  return inStrm->GetBytes(mData + thePos, inLen);
}

int StreamMemory::Seek(DWORD inPos)
{
  mPos = inPos;
  return  (NULL == mData) ? -1 : 0;
}

DWORD StreamMemory::Tell(void)
{
  return (NULL == mData) ? DWORD(-1) : mPos;
}

DWORD StreamMemory::GetLen(void)
{
  return (NULL == mData) ? DWORD(-1) : mDataLen;
}

int StreamMemory::IsEndOfStream(void)
{
  if (NULL == mData || mPos != mDataLen)
    return 0;
  else
    return 1;
}

Stream *StreamMemoryFactory::Create(void)
{
  return new StreamMemory;
}

} // namespace m_stream


// ----------------------------------------------------------------------------
//  eXgR[h
// ----------------------------------------------------------------------------

#ifdef TEST_M_STREAM_CPP

using namespace m_stream;

int main(int argc, char **argv)
{
  // --- StreamFile (t@Co) ̃eXg ---
  StreamFileFactory  theFactory;
  Stream  *theStream = theFactory.Create();

  if (0 != theStream->OpenIn(argv[1]))
  {
    printf("I[vo܂F%s\n", argv[1]);
    delete theStream;
    return 1;
  }

  int  theC;
  while (-1 != (theC = theStream->GetByte()))
    putchar(theC);

  delete theStream;

  return 0;
}

#endif // TEST_M_STREAM_CPP
